# Estimate empirical fdr given the files of
# 1. the p value from the real permutation
# 2. the p values from other permutations

# Example params from eQTLmap.sh
# ../qg/female.genvar.id female/eqtlMap femalePerm freeze2.185line.common.snp 0.05 100 24 female.empFDR.out

args <- commandArgs(TRUE)

# the list of all gene IDs, e.g. ../qg/female.genvar.id
gene.list <- args[1]

# the path and prefix to all primary (NOT permuted) plink output files, e.g. female/eqtlMap
obs.dir <- args[2]

# the path to all the concatenated perm results (see eQTLmap.sh comments), e.g. femalePerm
perm.dir <- args[3]

# the list of SNPs common to lines in use, e.g. freeze2.185line.common.snp
# This is generated by extracting just the 2nd column from the corresponding bim file
snp.list <- args[4]

# The FDR cutoff, e.g. 0.05
fdr <- as.numeric(args[5])

# The number of permutations, e.g. 100
n.perm <- as.numeric(args[6])

# The number of CPUs to use, e.g. 24
n.cpu <- as.numeric(args[7])

# The output file name, e.g. female.empFDR.out
out.file <- args[8]

# Parallelization setup
library(doMC)
registerDoMC(n.cpu)
sessionInfo()

gene.list <- scan(gene.list, what = "")
snp.list <- scan(snp.list, what = "")

fdr.out <- foreach (i = 1:length(gene.list), .combine = rbind) %dopar% {
    # first read the observed p value
    obs.pval <- read.table(paste(obs.dir, ".", gene.list[i], ".qassoc", sep = ""),
                                                    header = TRUE, as.is = TRUE)
      res <- c(gene.list[i], fdr, NA, NA)
      obs.pval <- obs.pval[obs.pval[, 2] %in% snp.list, ]
      if (nrow(obs.pval) > 0) {
            obs.pval <- obs.pval[order(obs.pval[, 9]), c(2, 9)]
                # read permuted data
                perm.pval <- data.frame(matrix(scan(paste(perm.dir, "/", gene.list[i], ".perm", sep = ""),
                                                                                            what = ""), ncol = 2, byrow = TRUE), stringsAsFactors = FALSE)
                perm.pval[, 2] <- as.numeric(perm.pval[, 2])
                perm.pval <- perm.pval[perm.pval[, 1] %in% snp.list, ]
                obs.pval <- cbind(obs.pval, rep(0, nrow(obs.pval)))
                for (j in 1:nrow(obs.pval)) {
                        obs.pval[j, 3] <- sum(perm.pval[, 2] <= obs.pval[j, 2])/n.perm/j
                      }
                # identify the cutoff point
                thres <- which(cummax(obs.pval[, 3]) < fdr)
                if (length(thres) > 0) {
                        res <- c(gene.list[i], obs.pval[max(thres), 3], paste(obs.pval[1:max(thres), 1], collapse = ","), paste(obs.pval[1:max(thres), 2], collapse = ","))
                      }
          }
      res
  }

write.table(fdr.out, file = out.file, sep = " ", quote = FALSE, row.names = FALSE, col.names = FALSE)
